import express from "express";
import dotenv from "dotenv";
import fetch from "node-fetch";
import fs from "fs";
import path from "path";
import { fileURLToPath } from "url";

const __filename = fileURLToPath(import.meta.url);
const __dirname = path.dirname(__filename);
const envPath = path.join(__dirname, ".env");
const exampleEnvPath = path.join(__dirname, "env.example");

if (fs.existsSync(envPath)) {
  dotenv.config({ path: envPath });
} else if (fs.existsSync(exampleEnvPath)) {
  console.warn(
    "[server] .env が見つかりませんでした。env.example を読み込んで起動します。"
  );
  dotenv.config({ path: exampleEnvPath });
} else {
  dotenv.config();
}

const app = express();
const PORT = process.env.PORT || 3001;
const ALLOWED_ORIGIN = process.env.CORS_ORIGIN || "http://localhost:5173";

app.use(express.json());

app.use((req, res, next) => {
  res.header("Access-Control-Allow-Origin", ALLOWED_ORIGIN);
  res.header(
    "Access-Control-Allow-Headers",
    "Origin, X-Requested-With, Content-Type, Accept, Authorization"
  );
  res.header("Access-Control-Allow-Methods", "GET, POST, OPTIONS");

  if (req.method === "OPTIONS") {
    return res.sendStatus(200);
  }

  next();
});

app.get("/health", (_, res) => {
  res.json({ status: "ok" });
});

app.get("/scribe-token", async (_req, res) => {
  if (!process.env.ELEVENLABS_API_KEY) {
    return res
      .status(500)
      .json({ error: "ELEVENLABS_API_KEY is not configured on the server." });
  }

  try {
    const response = await fetch(
      "https://api.elevenlabs.io/v1/single-use-token/realtime_scribe",
      {
        method: "POST",
        headers: {
          "xi-api-key": process.env.ELEVENLABS_API_KEY,
        },
      }
    );

    if (!response.ok) {
      const text = await response.text();
      console.error("Token API error:", response.status, text);
      return res.status(500).json({ error: "Failed to create token" });
    }

    const data = await response.json();
    res.json({ token: data.token });
  } catch (error) {
    console.error("Token endpoint error:", error);
    res.status(500).json({ error: "Internal server error" });
  }
});

app.listen(PORT, () => {
  console.log(`Server listening on http://localhost:${PORT}`);
});

